-- Z80 Emulator for ComputerCraft
local Z80 = {}

-- Registers
local registers = { A = 0, F = 0, B = 0, C = 0, D = 0, E = 0, H = 0, L = 0, SP = 0xFFFF, PC = 0x0000 }

-- Memory (64 KB)
local memory = {}; for i = 0, 0xFFFF do memory[i] = 0 end

-- Global Disk State
disk_state = {
    status_busy = false,
    status_ready = true,
    status_data_request = false,
    operation_in_progress = false,
    active_sector = 0,
    buffer = {},
    busy_timer = os.clock(),
}

-- Interrupt State
local interrupt_enabled = true; local interrupt_address = 0x0038

-- Disk Simulation: 32 KB (64 sectors of 512 bytes each)
local disk_file = "virtual_disk.img"
local sector_size = 512 -- 512 bytes per sector
local disk_size = 32 * 1024 -- 32 KB total
local sector_count = math.floor(disk_size / sector_size) -- Total sectors

local logged_errors = {} -- Store unique error messages
local log_file = "debug_log.txt" -- Path to the debug log file

-- Function to log messages
local function log_to_file(message)
    -- Deduplicate messages
    if logged_errors[message] then return end -- Skip if already logged
    logged_errors[message] = true -- Mark this message as logged

    -- Write to file
    local handle = fs.open(log_file, "a")
    if handle then
        handle.writeLine(os.date("%Y-%m-%d %H:%M:%S") .. " - " .. message)
        handle.close()
    else
        print("Failed to write to log file.")
    end
end


-- Initialize the disk file
local function initialize_disk()
    if not fs.exists(disk_file) then
        local handle = fs.open(disk_file, "wb")
        for i = 1, disk_size do handle.write(0x00) end
        handle.close()
        print("Disk initialized with 32 KB of space.")
    else
        print("Disk already exists.")
    end
end

-- Read a sector (returns 512 bytes)
local function read_sector(sector)
    if sector < 0 or sector >= sector_count then error("Sector out of bounds: " .. sector) end
    local handle = fs.open(disk_file, "rb")
    handle.seek("set", sector * sector_size)
    local data = {}
    for i = 1, sector_size do data[i] = handle.read() end
    handle.close()
    return data
end

-- Write a sector (takes 512 bytes)
local function write_sector(sector, data)
    if sector < 0 or sector >= sector_count then error("Sector out of bounds: " .. sector) end
    if #data ~= sector_size then error("Data size must be exactly " .. sector_size .. " bytes.") end
    local handle = fs.open(disk_file, "rb+")
    handle.seek("set", sector * sector_size)
    for i = 1, sector_size do handle.write(data[i]) end
    handle.close()
end

-- Storage Device Simulation
local storage_file = "storage_device.txt"; local storage = {}; for i = 0, 0xFFFF do storage[i] = 0 end
local function load_storage() local handle = fs.open(storage_file, "rb"); if handle then local i = 0; while true do local byte = handle.read(); if not byte then break end; storage[i] = byte; i = i + 1 end; handle.close() else print("No storage file found. Initializing empty storage.") end end
local function save_storage() local handle = fs.open(storage_file, "wb"); if handle then for i = 0, 0xFFFF do handle.write(storage[i] or 0) end; handle.close() else print("Failed to save storage.") end end

-- I/O Port Handling
local io_ports, last_address, buffer, active_sector = {}, 0, {}, 0

local function init_z80_ports()
    -- Disk Controller I/O Ports (0x10 to 0x17)
    for port = 0x10, 0x17 do
        io_ports[port] = function(value)
            if value == "update" then
                -- Handle state updates for disk
                if port == 0x17 and disk_state.status_busy and os.clock() > disk_state.busy_timer then
                    disk_state.status_busy = false
                    disk_state.status_ready = true
                    disk_state.status_data_request = true
                end
            elseif type(value) == "number" then
                -- Process port-specific logic
                if port == 0x10 then
                    disk_state.active_sector = bit32.band(disk_state.active_sector, 0xFF00) + value
                elseif port == 0x11 then
                    disk_state.active_sector = bit32.band(disk_state.active_sector, 0x00FF) + bit32.lshift(value, 8)
                elseif port == 0x12 then
                    disk_state.buffer = read_sector(disk_state.active_sector)
                    print(string.format("Read sector: %d", disk_state.active_sector))
                    disk_state.status_busy = true
                    disk_state.status_ready = false
                    disk_state.status_data_request = false
                    disk_state.busy_timer = os.clock() + 0.05 -- 50ms delay
                    disk_state.operation_in_progress = "read"
                elseif port == 0x13 then
                    write_sector(disk_state.active_sector, disk_state.buffer)
                    print(string.format("Write sector: %d", disk_state.active_sector))
                    disk_state.status_busy = true
                    disk_state.status_ready = false
                    disk_state.status_data_request = false
                    disk_state.busy_timer = os.clock() + 0.05 -- 50ms delay
                    disk_state.operation_in_progress = "write"
                elseif port == 0x17 then
                    -- Construct the status byte
                    local status = 0x00
                    if disk_state.status_busy then status = bit32.bor(status, 0x80) end -- BUSY
                    if disk_state.status_ready then status = bit32.bor(status, 0x40) end -- DRDY
                    if disk_state.status_data_request then status = bit32.bor(status, 0x08) end -- DRQ
                    print(string.format("Read from port 0x%02X (Status): 0x%02X", port, status))
                    return status
                end
            elseif value == nil then
                -- Gracefully handle nil values
                print(string.format("Port 0x%02X received nil value; ignoring operation.", port))
            else
                -- Invalid value type
                print(string.format("Invalid value type for port 0x%02X: %s", port, type(value)))
                error("Expected a number, 'update', or nil for I/O port operations.")
            end
        end
    end
end

    -- Disk Status Flags
    local status_busy = false
    local status_ready = true
    local status_data_request = false
    local operation_in_progress = false

    -- Timer for managing state transitions
    local busy_timer = os.clock()

    -- Disk Controller (Compact Flash) Enhanced Handling
    for port = 0x10, 0x17 do
        io_ports[port] = function(value)
            if port == 0x10 then
                -- Low byte of the active sector
                active_sector = bit32.band(active_sector, 0xFF00) + value
            elseif port == 0x11 then
                -- High byte of the active sector
                active_sector = bit32.band(active_sector, 0x00FF) + bit32.lshift(value, 8)
            elseif port == 0x12 then
                -- Trigger read sector
                buffer = read_sector(active_sector)
                print(string.format("Read sector: %d", active_sector))
                status_busy = true
                status_ready = false
                status_data_request = false
                busy_timer = os.clock() + 0.05 -- Simulate 50ms delay for operation
                operation_in_progress = "read"
            elseif port == 0x13 then
                -- Trigger write sector
                write_sector(active_sector, buffer)
                print(string.format("Write sector: %d", active_sector))
                status_busy = true
                status_ready = false
                status_data_request = false
                busy_timer = os.clock() + 0.05 -- Simulate 50ms delay for operation
                operation_in_progress = "write"
            elseif port == 0x17 then
                -- Simulate dynamic Status Register
                if status_busy and os.clock() > busy_timer then
                    -- Transition from BUSY to READY/DRQ
                    status_busy = false
                    status_ready = true
                    status_data_request = true -- DRQ set when operation is ready
                    print("DEBUG: Disk transitioned to READY with DRQ set.")
                end

                local status = 0x00
                if status_busy then status = bit32.bor(status, 0x80) end -- BUSY
                if status_ready then status = bit32.bor(status, 0x40) end -- DRDY
                if status_data_request then status = bit32.bor(status, 0x08) end -- DRQ
                print(string.format("Read from port 0x%02X (Status): 0x%02X", port, status))

                -- Clear DRQ after the status is read, if expected by the ROM
                if status_data_request then
                    status_data_request = false
                    print("DEBUG: DRQ cleared after status read.")
                end

                return status
            elseif port >= 0x14 and port <= 0x17 then
                local index = (port - 0x14) * 128
                if value then
                    -- Write to buffer (for write operations)
                    for i = 1, 128 do
                        buffer[index + i] = value
                    end
                    status_data_request = false -- Clear DRQ after data write
                else
                    -- Read from buffer (for read operations)
                    if operation_in_progress == "read" then
                        local return_value = buffer[index + 1] or 0x00
                        print(string.format("Read data from port 0x%02X: 0x%02X", port, return_value))
                        status_data_request = false -- Clear DRQ after data read
                        return return_value
                    else
                        print(string.format("Read from port 0x%02X: No data ready", port))
                        return 0x00
                    end
                end
            else
                -- Unhandled ports
                print(string.format("Unhandled disk I/O: Port=0x%02X, Value=0x%02X", port, value or 0))
            end
        end
    end

    -- Default handler for unused ports
    setmetatable(io_ports, {
        __index = function(_, port)
            return function(value)
                print(string.format("Unhandled I/O: Port=0x%02X, Value=0x%02X", port, value))
            end
        end
    })

    local function read_io_port(port)
        if io_ports[port] then
            local value = io_ports[port](nil) or 0xFF
            if value == nil then
                local log_message = string.format(
                    "DEBUG: Port 0x%02X returned nil value in read_io_port!\nRegisters: A=%02X, PC=%04X\nStack trace:\n%s",
                    port,
                    registers.A,
                    registers.PC,
                    debug.traceback()
                )
                log_to_file(log_message)
            end
            log_to_file(string.format("DEBUG: Port 0x%02X accessed: Value=0x%02X, PC=0x%04X", port, value or 0xFF, registers.PC))
            if port == 0x17 then
                log_to_file(string.format(
                    "DEBUG: Disk State - busy=%s, ready=%s, data_request=%s",
                    tostring(disk_state.status_busy),
                    tostring(disk_state.status_ready),
                    tostring(disk_state.status_data_request)
                ))
            end
            return value
        else
            local log_message = string.format("Read from unmapped port 0x%02X, returning 0xFF", port)
            log_to_file(log_message)
            return 0xFF
        end
    end       

    local function write_io_port(port, value)
        if value == nil then
            local log_message = string.format(
                "DEBUG: Port 0x%02X received nil value in write_io_port!\nRegisters: A=%02X, PC=%04X\nStack trace:\n%s",
                port,
                registers.A,
                registers.PC,
                debug.traceback()
            )
            log_to_file(log_message)
            return
        end
        if io_ports[port] then
            io_ports[port](value)
        else
            local log_message = string.format("Write to unmapped port 0x%02X, Value=0x%02X", port, value or 0xFF)
            log_to_file(log_message)
        end
    end         

-- Instruction Dispatch Tables
local instructions, cb_instructions = {}, {}

-- Helpers for Memory Access
local function read_memory(address) return memory[bit32.band(address, 0xFFFF)] or 0 end
local function write_memory(address, value) memory[bit32.band(address, 0xFFFF)] = bit32.band(value, 0xFF) end

-- Helpers for Flags
local function set_flags(zero, sign, half_carry, carry) registers.F = 0; if zero then registers.F = bit32.bor(registers.F, 0x40) end; if sign then registers.F = bit32.bor(registers.F, 0x80) end; if half_carry then registers.F = bit32.bor(registers.F, 0x10) end; if carry then registers.F = bit32.bor(registers.F, 0x01) end end
local function get_flags() return { zero = (bit32.band(registers.F, 0x40) ~= 0), sign = (bit32.band(registers.F, 0x80) ~= 0), half_carry = (bit32.band(registers.F, 0x10) ~= 0), carry = (bit32.band(registers.F, 0x01) ~= 0) } end

-- Interrupt Handling
local function handle_interrupt() if interrupt_enabled then write_memory(registers.SP - 1, bit32.band(bit32.rshift(registers.PC, 8), 0xFF)); write_memory(registers.SP - 2, bit32.band(registers.PC, 0xFF)); write_memory(registers.SP - 3, registers.F); registers.SP = bit32.band(registers.SP - 3, 0xFFFF); registers.PC = interrupt_address; interrupt_enabled = false end end

-- Register Table for Dynamic Instruction Generation
local reg_map = { "B", "C", "D", "E", "H", "L", "(HL)", "A" }
for dest = 1, #reg_map do for src = 1, #reg_map do local opcode = 0x40 + (dest - 1) * 8 + (src - 1); instructions[opcode] = function() local value; if reg_map[src] == "(HL)" then local address = bit32.bor(bit32.lshift(registers.H, 8), registers.L); value = read_memory(address) else value = registers[reg_map[src]] end; if reg_map[dest] == "(HL)" then local address = bit32.bor(bit32.lshift(registers.H, 8), registers.L); write_memory(address, value) else registers[reg_map[dest]] = value end end end end

-- Basic Instructions
instructions[0x00] = function() end; instructions[0xC3] = function() local low = read_memory(registers.PC); registers.PC = bit32.band(registers.PC + 1, 0xFFFF); local high = read_memory(registers.PC); registers.PC = bit32.band(registers.PC + 1, 0xFFFF); registers.PC = bit32.bor(low, bit32.lshift(high, 8)) end; instructions[0x3E] = function() registers.A = read_memory(registers.PC); registers.PC = bit32.band(registers.PC + 1, 0xFFFF) end; instructions[0x06] = function() registers.B = read_memory(registers.PC); registers.PC = bit32.band(registers.PC + 1, 0xFFFF) end; instructions[0x76] = function() error("HALT instruction encountered.") end; instructions[0xF3] = function() interrupt_enabled = false; print("DI executed: Interrupts disabled.") end; instructions[0xFB] = function() interrupt_enabled = true; print("EI executed: Interrupts enabled.") end; instructions[0x31] = function() local low = read_memory(registers.PC); registers.PC = bit32.band(registers.PC + 1, 0xFFFF); local high = read_memory(registers.PC); registers.PC = bit32.band(registers.PC + 1, 0xFFFF); registers.SP = bit32.bor(low, bit32.lshift(high, 8)) end; instructions[0xD3] = function() local port = read_memory(registers.PC); registers.PC = bit32.band(registers.PC + 1, 0xFFFF); write_io_port(port, registers.A) end; instructions[0xDB] = function() local port = read_memory(registers.PC); registers.PC = bit32.band(registers.PC + 1, 0xFFFF); registers.A = read_io_port(port) end; instructions[0xAF] = function() registers.A = bit32.band(bit32.bxor(registers.A, registers.A), 0xFF); set_flags(registers.A == 0, false, false, false) end; instructions[0xCD] = function() local low = read_memory(registers.PC); registers.PC = bit32.band(registers.PC + 1, 0xFFFF); local high = read_memory(registers.PC); registers.PC = bit32.band(registers.PC + 1, 0xFFFF); local addr = bit32.bor(low, bit32.lshift(high, 8)); write_memory(registers.SP - 1, bit32.band(bit32.rshift(registers.PC, 8), 0xFF)); write_memory(registers.SP - 2, bit32.band(registers.PC, 0xFF)); registers.SP = bit32.band(registers.SP - 2, 0xFFFF); registers.PC = addr end; instructions[0xCB] = function() local opcode = read_memory(registers.PC); registers.PC = bit32.band(registers.PC + 1, 0xFFFF); if cb_instructions[opcode] then cb_instructions[opcode]() else error(string.format("Unsupported CB-prefixed opcode: 0x%02X", opcode)) end end

-- CB-Prefixed Instructions
cb_instructions[0x77] = function() local address = bit32.bor(bit32.lshift(registers.H, 8), registers.L); local bit_pos = 6; local value = read_memory(address); print(string.format("BIT 6, (HL): Address=0x%04X, Value=0x%02X", address, value)); if bit32.band(value, bit32.lshift(1, bit_pos)) ~= 0 then set_flags(true, false, true, false) else set_flags(false, false, true, false) end end
instructions[0x28] = function() local offset = read_memory(registers.PC); registers.PC = bit32.band(registers.PC + 1, 0xFFFF); if get_flags().zero then if offset > 127 then offset = offset - 256 end; registers.PC = bit32.band(registers.PC + offset, 0xFFFF) end end

-- Fetch-Decode-Execute Cycle
local instruction_count = 0;local function step()
    -- Handle any interrupts
    handle_interrupt()

    -- Dynamically update port states (e.g., disk, CTC, or RTC status)
    for port, handler in pairs(io_ports) do
        if type(handler) == "function" then
            handler("update") -- Trigger state updates
        end
    end

    -- Fetch and execute the next instruction
    local opcode = read_memory(registers.PC)
    print(string.format("Instruction executed: 0x%02X, PC=0x%04X", opcode, registers.PC))
    registers.PC = bit32.band(registers.PC + 1, 0xFFFF)
    local instruction = instructions[opcode]
    if instruction then
        instruction()
    else
        print(string.format("Unsupported opcode: 0x%02X at PC=0x%04X", opcode, registers.PC - 1))
        error("Execution halted due to unsupported opcode.")
    end

    -- Yield periodically to prevent freezing the ComputerCraft environment
    instruction_count = instruction_count + 1
    if instruction_count % 1000 == 0 then
        os.queueEvent("yield")
        os.pullEvent("yield")
    end
end

-- Load ROM into Memory
local function load_rom(file) local handle = fs.open(file, "rb"); if not handle then error("Failed to open ROM file.") end; local address = 0x0000; while true do local byte = handle.read(); if not byte then break end; write_memory(address, byte); address = address + 1 end; handle.close(); print("ROM loaded successfully.") end

-- Main Entry Point
local function main()
    print("Z80 Emulator v1.0")
    initialize_disk()         -- Initialize the disk file
    init_z80_ports()          -- Initialize Z80 ports
    load_rom("fuzix.rom")     -- Load the ROM file
    load_storage()            -- Load persistent storage
    print("Running...")
    local success, err = pcall(function()
        while true do
            step()
        end
    end)
    save_storage()            -- Save storage on exit
    if not success then
        print("Execution stopped:", err)
    end
end

-- Start the emulator
main()